package coreservlets.filters;

import java.io.*;
import java.util.zip.*;
import javax.servlet.*;
import javax.servlet.http.*;

/** Filtr kompresuje dane wynikowe przy uyciu gzip
 *  (przy zaozeniu, e przegldarka obsuguje kompresj gzip).
 *  <P>
 *  Taken from Core Servlets and JavaServer Pages Volume II
 *  from Prentice Hall and Sun Microsystems Press,
 *  http://volume2.coreservlets.com/.
 *  (C) 2007 Marty Hall, Larry Brown, and Yaakov Chaikin;
 *  may be freely used or adapted.
 */
public class CompressionFilter implements Filter {
  private FilterConfig config;

  /** Jeli przegldarka nie obsuguje kompresji gzip,
   *  zasb wywoywany jest w standardowy sposb. Jeli
   *  przegldarka <I>obsuguje</I> kompresj gzip,
   *  definiowany nagwek odpowiedzi Content-Encoding
   *  i nastpuje wywoanie zasobu z odpowiedzi w obiekcie
   *  przechowujcym, ktry gromadzi wszystkie dane wynikowe
   *  i zapisuje w tablicy skompresowanych bajtw.
   *  Na kocu tablica ta zostaje zapisana
   *  w strumieniu danych wynikowych dla klienta.
   */
  public void doFilter(ServletRequest request,
                       ServletResponse response,
                       FilterChain chain)
      throws ServletException, IOException {
    HttpServletRequest req = (HttpServletRequest)request;
    HttpServletResponse res = (HttpServletResponse)response;
    if (!isGzipSupported(req)) {
      // Standardowe wywoanie zasobu.
      chain.doFilter(req,res);
    } else {
      // Informacja dla przegldarki,
      // e wysyane s dane skompresowane gzip.
      res.setHeader("Content-Encoding", "gzip");
      
      // Wywoanie zasobu, zebranie danych wynikowych
      // w obiekcie przechowujcym.
      StringWrapper responseWrapper =
        new StringWrapper(res);
      chain.doFilter(req,responseWrapper);

      // Utworzenie obiektu kompresujcego dane
      // i umieszczenie ich w tablicy bajtw.
      ByteArrayOutputStream byteStream =
        new ByteArrayOutputStream();
      GZIPOutputStream zipOut =
        new GZIPOutputStream(byteStream);
      OutputStreamWriter tempOut =
        new OutputStreamWriter(zipOut);
      
      // Kompresja oryginalnych danych wynikowych
      // i umieszczenie ich w tablicy bajtw.
      tempOut.write(responseWrapper.toString());
      
      // Strumienie gzip trzeba jawnie zamkn.
      tempOut.close();

      // Wysanie skompresowanych danych do klienta.
      OutputStream realOut = res.getOutputStream();
      byteStream.writeTo(realOut);
    }
  }

  /** Zapisanie obiektu FilterConfig na wypadek,
   *  gdyby potrzeboway ich klasy potomne.
   */
  public void init(FilterConfig config)
      throws ServletException {
    this.config = config;
  }

  protected FilterConfig getFilterConfig() {
    return(config);
  }
  
  public void destroy() {}

  private boolean isGzipSupported(HttpServletRequest req) {
    String browserEncodings =
      req.getHeader("Accept-Encoding");
    return((browserEncodings != null) &&
           (browserEncodings.indexOf("gzip") != -1));
  }
}
